/***
 * Copyright (C) 2010 Johan Henriksson
 * This code is under the Endrov / BSD license. See www.endrov.net
 * for the full text and how to cite.
 */
package endrov.data;

import java.util.*;

import javax.swing.Icon;

import org.jdom.Element;

import endrov.core.EndrovCore;
import endrov.core.EndrovUtil;
import endrov.core.log.EvLog;
import endrov.data.basicTypes.EvCustomObject;
import endrov.gui.icon.BasicIcon;
import endrov.util.math.EvDecimal;


/**
 * EV Object container
 * @author Johan Henriksson
 */
public class EvContainer
	{
	/** All meta objects */
	public TreeMap<String,EvObject> metaObject=new TreeMap<String,EvObject>();

	/**
	 * Icon representing this container or data type. Should be set by subtype
	 * TODO better, provide a method. need class EvContainerEmpty
	 */
	public Icon containerIcon=BasicIcon.iconData;
	
	/**
	 * Is the data autogenerated? this hints many things:
	 * * If autogenerated, it need not be saved, it can be autogenerated again
	 * * If not autogenerated, then it doesn't make sense to overwrite it with autogenerated data (saves the user from mistakes)
	 */
	public boolean isGeneratedData=false;
	
	/**
	 * Get one child
	 */
	public EvObject getChild(String name)
		{
		return metaObject.get(name);
		}
	
	/*
	public EvContainer getChild(EvPath path)
		{
		EvContainer c=this;
		for(String s:path.path)
			{
			c=c.getChild(s);
			if(c==null)
				return null; //Or throw an exception?
			}
		return c;
		}*/
	
	/**
	 * Get the names of all children
	 */
	public Set<String> getChildNames()
		{
		return metaObject.keySet();
		}
	
	//TODO should make accessors
	//TODO parent pointer?
	//TODO putting an object should be checked to not create a cycle
	
	
	/** Flag if the metadata container itself has been modified */
	public boolean coreMetadataModified=false;
	
	/**
	 * This blobID is only valid *exactly after XML has been read*. This is because is really should
	 * be stored in the I/O-session, but more conveniently located here since 
	 */
	public String ostBlobID;
	
	/**
	 * Date when object was created. Can be null. Unix time
	 */
	public EvDecimal dateCreate;
		
	/**
	 * Date when this object was last modified. Can be null. Unix time.
	 */
	public EvDecimal dateLastModify;
	
	/**
	 * Who created the object. Can be null
	 */
	public String author;

	
	public EvContainer()
		{
		dateCreate=new EvDecimal(System.currentTimeMillis());
		}
	
	/**
	 * Set that metadata is not modified (recursively). This is done
	 * after the data has been stored.
	 */
	public void setMetadataNotModified()
		{
		coreMetadataModified=false;
		for(EvContainer ob:metaObject.values())
			ob.setMetadataNotModified();
		}
	
	/**
	 * State that metadata has been modified
	 */
	public void setMetadataModified()
		{
		coreMetadataModified=true;
		dateLastModify=new EvDecimal(System.currentTimeMillis());
		}
	
	/**
	 * Check if the metadata or any object has been modified
	 */
	public boolean isMetadataModified()
		{
		boolean modified=coreMetadataModified;
		for(EvObject ob:metaObject.values())
			modified|=ob.isMetadataModified();
		return modified;
		}
	

	/**
	 * Get all objects of a certain type
	 */
	@SuppressWarnings("unchecked") public <E> List<E> getObjects(Class<E> cl)
		{
		LinkedList<E> ll=new LinkedList<E>();
		for(EvObject ob2:metaObject.values())
			if(cl.isInstance(ob2))
				ll.add((E)ob2);
		return ll;
		}
	
	/**
	 * Get all ID and objects of a certain type
	 */
	@SuppressWarnings("unchecked") public <E> SortedMap<String, E> getIdObjects(Class<E> cl)
		{
		TreeMap<String, E> map=new TreeMap<String, E>();
		for(Map.Entry<String, EvObject> e:metaObject.entrySet())
			if(cl.isInstance(e.getValue()))
				map.put(e.getKey(),(E)e.getValue());
		return map;
		}

	/**
	 * Get all ID and objects of a certain type
	 */
	public <E> SortedMap<EvPath, E> getIdObjectsRecursive(Class<E> cl)
		{
		TreeMap<EvPath, E> map=new TreeMap<EvPath, E>();
		getIdObjectsRecursiveHelper(this, map, new LinkedList<String>(), cl);
		return map;
		}
	@SuppressWarnings("unchecked") private <E> void getIdObjectsRecursiveHelper(EvContainer root, Map<EvPath, E> map, LinkedList<String> curPath, Class<E> cl)
		{
		for(Map.Entry<String, EvObject> e:metaObject.entrySet())
			{
			curPath.addLast(e.getKey());
			if(cl.isInstance(e.getValue()))
				map.put(new EvPath(root, curPath),(E)e.getValue());
			((EvContainer)e.getValue()).getIdObjectsRecursiveHelper(root, map, curPath, cl);
			curPath.removeLast();
			}
		}
	
	
	/**
	 * Get a meta object by ID
	 */
	public EvObject getMetaObject(String i)
		{
		return metaObject.get(i);
		}
	
	public EvContainer getMetaContainer(String... name)
		{
		return getMetaContainer(0, name);
		}
	private EvContainer getMetaContainer(int index,String... name)
		{
		if(index==name.length)
			return this;
		else
			return ((EvContainer)metaObject.get(name[index])).getMetaContainer(index+1, name);
		}
	/**
	 * Put a meta object into the collection
	 */
	public String addMetaObject(EvObject o)
		{
		String id=getFreeChildName();
		metaObject.put(id, o);
		return id;
		}
	
	public String getFreeChildName()
		{
		int i=1;
		while(metaObject.get(Integer.toString(i))!=null)
			i++;
		String id=Integer.toString(i);
		return id;
		}
	
	
	/**
	 * Remove an object via the pointer
	 */
	public void removeMetaObjectByValue(EvContainer ob)
		{
		String id=null;
		for(Map.Entry<String, EvObject> entry:metaObject.entrySet())
			if(entry.getValue()==ob)
				{
				id=entry.getKey();
				break;
				}
		if(id!=null)
			metaObject.remove(id);
		}	
	
	
	private static final String tagChild="_ostchild";
	private final static String tempString="__TEMPNAME__";
	
	/**
	 * Serialize object and all children
	 */
	void recursiveSaveMetadata(Element root)
		{
		for(String id:metaObject.keySet())
			{
			EvObject o=metaObject.get(id);
			if(!o.isGeneratedData) //Generated data is never saved
				{
				Element el=new Element(tempString);
				el.setAttribute("id",""+id);
				if(o.ostBlobID!=null)
					el.setAttribute("ostblobid",o.ostBlobID);
				if(o.dateCreate!=null)
					el.setAttribute("ostdatecreate",o.dateCreate.toString());
				if(o.dateLastModify!=null)
					el.setAttribute("ostdatemodify",o.dateLastModify.toString());
				if(o.author!=null)
					el.setAttribute("ostauthor",o.author);
				
				Element eData=new Element("data");
				String metatypeName=o.saveMetadata(eData);
				if(metatypeName==null)
					throw new RuntimeException("Object of type "+o.getClass()+" reports null metadata name");
				el.setName(metatypeName);
				el.addContent(eData);
				
				if(el.getName().equals(tempString))
					throw new RuntimeException("Plugin for "+o.getClass()+" does not save properly");
				
				//also save subobjects
				if(!o.metaObject.isEmpty())
					{
					Element sube=new Element(tagChild);
					o.recursiveSaveMetadata(sube);
					el.addContent(sube);
					}
				root.addContent(el);
				}
			else
				System.out.println("Skipping saving metadata "+id);
			}
		}

	/**
	 * Load object and all children
	 */
	void recursiveLoadMetadata(Element element)
		{
		//Extract objects
		for(Element child:EndrovUtil.castIterableElement(element.getChildren()))
			{			
			String childName=child.getName();
			Class<? extends EvObject> ext=EvData.supportedMetadataFormats.get(childName);
			EvObject o=null;
			if(ext==null)
				{
				o=new EvCustomObject();
				EvLog.printLog("Warning: "+childName+" was not detected to be of a known type");
				}
			else
				{
				try
					{
					o=ext.newInstance();
					}
				catch (InstantiationException e)
					{
					e.printStackTrace();
					}
				catch (IllegalAccessException e)
					{
					e.printStackTrace();
					}
				}
			String id=child.getAttributeValue("id");
			
			//Common data for all OST objects
			o.ostBlobID=child.getAttributeValue("ostblobid");
			
			String dateCreate=child.getAttributeValue("ostdatecreate");
			if(dateCreate!=null)
				o.dateCreate=new EvDecimal(dateCreate);
			else
				o.dateCreate=null;
			String dateModify=child.getAttributeValue("ostdatemodify");
			if(dateModify!=null)
				o.dateLastModify=new EvDecimal(dateModify);
			o.author=child.getAttributeValue("ostauthor");
			
			
			Element subob=child.getChild(tagChild); 
			if(subob!=null)
				{
				o.recursiveLoadMetadata(subob);
				child.removeContent(subob);
				}
			o.loadMetadata(child.getChild("data"));

			metaObject.put(id, o);
			if(EndrovCore.debugMode)
				EvLog.printLog("Found meta object of type "+child.getName());
			}
		}
	
	
	/**
	 * Extract EvObjects from an element.
	 */
	/*
	private static Map<String,EvObject> extractSubObjectsFromXML(Element element)
		{
		Map<String,EvObject> obs=new HashMap<String, EvObject>();
	//Extract objects
		for(Element child:EV.castIterableElement(element.getChildren()))
			{
			Class<? extends EvObject> ext=EvData.extensions.get(child.getName());
			EvObject o=null;
			if(ext==null)
				{
				o=new CustomObject();
				o.loadMetadata(child);
				Log.printLog("Found unknown meta object of type "+child.getName());
				}
			else
				{
				try
					{
					o=ext.newInstance();
					}
				catch (InstantiationException e)
					{
					e.printStackTrace();
					}
				catch (IllegalAccessException e)
					{
					e.printStackTrace();
					}
				o.loadMetadata(child);
				Log.printLog("Found meta object of type "+child.getName());
				}
			String sid=child.getAttributeValue("id");
			String id;
			if(sid==null) 
				//This is only needed for imagesets without the EV extended attributes
				//should maybe grab a free one (detect)
				//id=""+-1;
				id="im"; //This is for the OST3 transition
			else
				id=sid;
			obs.put(id, o);
			}
		return obs;
		}
*/
	
	
	public Icon getContainerIcon()
		{
		return containerIcon;
		}

	public void putChild(String name, EvObject child)
		{
		metaObject.put(name, child);
		}
	
	}
