/*
 * Decompiled with CFR 0.152.
 */
package loci.formats.in;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.zip.GZIPInputStream;
import loci.common.DateTools;
import loci.common.Location;
import loci.common.RandomAccessInputStream;
import loci.formats.FormatException;
import loci.formats.FormatReader;
import loci.formats.FormatTools;
import loci.formats.MetadataTools;
import loci.formats.in.MetadataLevel;
import loci.formats.meta.MetadataStore;
import ome.xml.model.primitives.PositiveFloat;
import ome.xml.model.primitives.PositiveInteger;

public class ICSReader
extends FormatReader {
    public static final String NL = "\r\n";
    public static final String[] DATE_FORMATS = new String[]{"EEEE, MMMM dd, yyyy HH:mm:ss", "EEE dd MMMM yyyy HH:mm:ss", "EEE MMM dd HH:mm:ss yyyy", "EE dd MMM yyyy HH:mm:ss z", "HH:mm:ss dd\\MM\\yy"};
    private static final String[][] DOCUMENT_KEYS = new String[][]{{"date"}, {"document", "average"}, {"document"}, {"gmtdate"}, {"label"}};
    private static final String[][] HISTORY_KEYS = new String[][]{{"a\\d"}, {"acquisition", "acquire\\..*."}, {"acquisition", "laserbox\\..*."}, {"acquisition", "modules\\(.*."}, {"acquisition", "objective", "position"}, {"adc", "resolution"}, {"atd_hardware", "ver"}, {"atd_libraries", "ver"}, {"atd_microscopy", "ver"}, {"author"}, {"averagecount"}, {"averagequality"}, {"beam", "zoom"}, {"binning"}, {"bits/pixel"}, {"black", "level"}, {"black", "level\\*"}, {"black_level"}, {"camera", "manufacturer"}, {"camera", "model"}, {"camera"}, {"cfd", "holdoff"}, {"cfd", "limit", "high"}, {"cfd", "limit", "low"}, {"cfd", "zc", "level"}, {"channel\\*"}, {"collection", "time"}, {"cols"}, {"company"}, {"count", "increment"}, {"created", "on"}, {"creation", "date"}, {"cube", "descriptio"}, {"cube", "description"}, {"cube", "emm", "nm"}, {"cube", "exc", "nm"}, {"cube"}, {"date"}, {"dategmt"}, {"dead", "time", "comp"}, {"desc", "exc", "turret"}, {"desc", "emm", "turret"}, {"detector", "type"}, {"detector"}, {"dimensions"}, {"direct", "turret"}, {"dither", "range"}, {"dwell"}, {"excitationfwhm"}, {"experiment"}, {"experimenter"}, {"expon.", "order"}, {"exposure"}, {"exposure_time"}, {"ext", "latch", "delay"}, {"extents"}, {"filterset", "dichroic", "name"}, {"filterset", "dichroic", "nm"}, {"filterset", "emm", "name"}, {"filterset", "emm", "nm"}, {"filterset", "exc", "name"}, {"filterset", "exc", "nm"}, {"filterset"}, {"filter\\*"}, {"firmware"}, {"fret", "backgr\\d"}, {"frametime"}, {"gain"}, {"gain\\d"}, {"gain\\*"}, {"gamma"}, {"icsviewer", "ver"}, {"ht\\*"}, {"id"}, {"illumination", "mode", "laser"}, {"illumination", "mode"}, {"image", "bigendian"}, {"image", "bpp"}, {"image", "form"}, {"image", "physical_sizex"}, {"image", "physical_sizey"}, {"image", "sizex"}, {"image", "sizey"}, {"labels"}, {"lamp", "manufacturer"}, {"lamp", "model"}, {"laser", "firmware"}, {"laser", "manufacturer"}, {"laser", "model"}, {"laser", "power"}, {"laser", "rep", "rate"}, {"laser", "type"}, {"laser\\d", "intensity"}, {"laser\\d", "name"}, {"laser\\d", "wavelength"}, {"left"}, {"length"}, {"line", "compressio"}, {"line", "compression"}, {"linetime"}, {"magnification"}, {"manufacturer"}, {"max", "photon", "coun"}, {"max", "photon", "count"}, {"memory", "bank"}, {"metadata", "format", "ver"}, {"microscope", "built", "on"}, {"microscope", "name"}, {"microscope"}, {"mirror", "\\d"}, {"mode"}, {"noiseval"}, {"no.", "frames"}, {"objective", "detail"}, {"objective", "immersion"}, {"objective", "mag"}, {"objective", "magnification"}, {"objective", "na"}, {"objective", "type"}, {"objective", "workingdistance"}, {"objective"}, {"offsets"}, {"other", "text"}, {"passcount"}, {"pinhole"}, {"pixel", "clock"}, {"pixel", "time"}, {"pmt"}, {"polarity"}, {"region"}, {"rep", "period"}, {"repeat", "time"}, {"revision"}, {"routing", "chan", "x"}, {"routing", "chan", "y"}, {"rows"}, {"scan", "borders"}, {"scan", "flyback"}, {"scan", "pattern"}, {"scan", "pixels", "x"}, {"scan", "pixels", "y"}, {"scan", "pos", "x"}, {"scan", "pos", "y"}, {"scan", "resolution"}, {"scan", "speed"}, {"scan", "zoom"}, {"scanner", "lag"}, {"scanner", "pixel", "time"}, {"scanner", "resolution"}, {"scanner", "speed"}, {"scanner", "xshift"}, {"scanner", "yshift"}, {"scanner", "zoom"}, {"shutter\\d"}, {"shutter", "type"}, {"software"}, {"spectral", "bin_definition"}, {"spectral", "calibration", "gain", "data"}, {"spectral", "calibration", "gain", "mode"}, {"spectral", "calibration", "offset", "data"}, {"spectral", "calibration", "offset", "mode"}, {"spectral", "calibration", "sensitivity", "mode"}, {"spectral", "central_wavelength"}, {"spectral", "laser_shield"}, {"spectral", "laser_shield_width"}, {"spectral", "resolution"}, {"stage", "controller"}, {"stage", "firmware"}, {"stage", "manufacturer"}, {"stage", "model"}, {"stage", "pos"}, {"stage", "positionx"}, {"stage", "positiony"}, {"stage", "positionz"}, {"stage_xyzum"}, {"step\\d", "channel", "\\d"}, {"step\\d", "gain", "\\d"}, {"step\\d", "laser"}, {"step\\d", "name"}, {"step\\d", "pinhole"}, {"step\\d", "pmt", "ch", "\\d"}, {"step\\d", "shutter", "\\d"}, {"step\\d"}, {"stop", "on", "o'flow"}, {"stop", "on", "time"}, {"study"}, {"sync", "freq", "div"}, {"sync", "holdoff"}, {"sync"}, {"tac", "gain"}, {"tac", "limit", "low"}, {"tac", "offset"}, {"tac", "range"}, {"tau\\d"}, {"tcspc", "adc", "res"}, {"tcspc", "adc", "resolution"}, {"tcspc", "approx", "adc", "rate"}, {"tcspc", "approx", "cfd", "rate"}, {"tcspc", "approx", "tac", "rate"}, {"tcspc", "bh"}, {"tcspc", "cfd", "holdoff"}, {"tcspc", "cfd", "limit", "high"}, {"tcspc", "cfd", "limit", "low"}, {"tcspc", "cfd", "zc", "level"}, {"tcspc", "clock", "polarity"}, {"tcspc", "collection", "time"}, {"tcspc", "count", "increment"}, {"tcspc", "dead", "time", "enabled"}, {"tcspc", "delay"}, {"tcspc", "dither", "range"}, {"tcspc", "left", "border"}, {"tcspc", "line", "compression"}, {"tcspc", "mem", "offset"}, {"tcspc", "operation", "mode"}, {"tcspc", "overflow"}, {"tcspc", "pixel", "clk", "divider"}, {"tcspc", "pixel", "clock"}, {"tcspc", "routing", "x"}, {"tcspc", "routing", "y"}, {"tcspc", "scan", "x"}, {"tcspc", "scan", "y"}, {"tcspc", "sync", "divider"}, {"tcspc", "sync", "holdoff"}, {"tcspc", "sync", "rate"}, {"tcspc", "sync", "threshold"}, {"tcspc", "sync", "zc", "level"}, {"tcspc", "tac", "gain"}, {"tcspc", "tac", "limit", "high"}, {"tcspc", "tac", "limit", "low"}, {"tcspc", "tac", "offset"}, {"tcspc", "tac", "range"}, {"tcspc", "time", "window"}, {"tcspc", "top", "border"}, {"tcspc", "total", "frames"}, {"tcspc", "total", "time"}, {"tcspc", "trigger"}, {"tcspc", "x", "sync", "polarity"}, {"tcspc", "y", "sync", "polarity"}, {"text"}, {"time"}, {"title"}, {"top"}, {"transmission"}, {"trigger"}, {"type"}, {"units"}, {"version"}, {"wavelength\\*"}, {"x", "amplitude"}, {"y", "amplitude"}, {"x", "delay"}, {"y", "delay"}, {"x", "offset"}, {"y", "offset"}, {"z", "\\(background\\)"}};
    private static final String[][] LAYOUT_KEYS = new String[][]{{"coordinates"}, {"order"}, {"parameters"}, {"real_significant_bits"}, {"significant_bits"}, {"significant_channels"}, {"sizes"}};
    private static final String[][] PARAMETER_KEYS = new String[][]{{"allowedlinemodes"}, {"ch"}, {"higher_limit"}, {"labels"}, {"lower_limit"}, {"origin"}, {"range"}, {"sample_width", "ch"}, {"sample_width"}, {"scale"}, {"units", "adc-units", "channels"}, {"units", "adc-units", "nm"}, {"units"}};
    private static final String[][] REPRESENTATION_KEYS = new String[][]{{"byte_order"}, {"compression"}, {"format"}, {"sign"}};
    private static final String[][] SENSOR_KEYS = new String[][]{{"model"}, {"s_params", "channels"}, {"s_params", "exphotoncnt"}, {"s_params", "lambdaem"}, {"s_params", "lambdaex"}, {"s_params", "numaperture"}, {"s_params", "pinholeradius"}, {"s_params", "pinholespacing"}, {"s_params", "refinxlensmedium"}, {"s_params", "refinxmedium"}, {"s_params", "refrinxlensmedium"}, {"s_params", "refrinxmedium"}, {"type"}};
    private static final String[][] VIEW_KEYS = new String[][]{{"view", "color", "lib", "lut"}, {"view", "color", "count"}, {"view", "color", "doc", "scale"}, {"view", "color", "mode", "rgb", "set"}, {"view", "color", "mode", "rgb"}, {"view", "color", "schemes"}, {"view", "color", "view", "active"}, {"view", "color"}, {"view\\d", "alpha"}, {"view\\d", "alphastate"}, {"view\\d", "annotation", "annellipse"}, {"view\\d", "annotation", "annpoint"}, {"view\\d", "autoresize"}, {"view\\d", "axis"}, {"view\\d", "blacklevel"}, {"view\\d", "color"}, {"view\\d", "cursor"}, {"view\\d", "dimviewoption"}, {"view\\d", "gamma"}, {"view\\d", "ignoreaspect"}, {"view\\d", "intzoom"}, {"view\\d", "live"}, {"view\\d", "order"}, {"view\\d", "port"}, {"view\\d", "position"}, {"view\\d", "saturation"}, {"view\\d", "scale"}, {"view\\d", "showall"}, {"view\\d", "showcursor"}, {"view\\d", "showindex"}, {"view\\d", "size"}, {"view\\d", "synchronize"}, {"view\\d", "tile"}, {"view\\d", "useunits"}, {"view\\d", "zoom"}, {"view\\d"}, {"view"}};
    private static String[][] OTHER_KEYS = new String[][]{{"cube", "descriptio"}, {"cube", "description"}, {"image", "form"}, {"refinxlensmedium"}, {"refinxmedium"}, {"scil_type"}, {"source"}};
    private static final String[] CATEGORIES = new String[]{"ics_version", "filename", "source", "layout", "representation", "parameter", "sensor", "history", "document", "view.*", "end", "file", "offset", "parameters", "order", "sizes", "coordinates", "significant_bits", "format", "sign", "compression", "byte_order", "origin", "scale", "units", "t", "labels", "SCIL_TYPE", "type", "model", "s_params", "gain.*", "dwell", "shutter.*", "pinhole", "laser.*", "version", "objective", "PassCount", "step.*", "date", "GMTdate", "label", "software", "author", "length", "Z (background)", "dimensions", "rep period", "image form", "extents", "offsets", "region", "expon. order", "a.*", "tau.*", "noiseval", "excitationfwhm", "created on", "text", "other text", "mode", "CFD limit low", "CFD limit high", "CFD zc level", "CFD holdoff", "SYNC zc level", "SYNC freq div", "SYNC holdoff", "TAC range", "TAC gain", "TAC offset", "TAC limit low", "ADC resolution", "Ext latch delay", "collection time", "repeat time", "stop on time", "stop on O'flow", "dither range", "count increment", "memory bank", "sync threshold", "dead time comp", "polarity", "line compressio", "scan flyback", "scan borders", "pixel time", "pixel clock", "trigger", "scan pixels x", "scan pixels y", "routing chan x", "routing chan y", "detector type", "channel.*", "filter.*", "wavelength.*", "black.level.*", "ht.*", "scan resolution", "scan speed", "scan zoom", "scan pattern", "scan pos x", "scan pos y", "transmission", "x amplitude", "y amplitude", "x offset", "y offset", "x delay", "y delay", "beam zoom", "mirror .*", "direct turret", "desc exc turret", "desc emm turret", "cube", "Stage_XYZum", "cube descriptio", "camera", "exposure", "bits/pixel", "binning", "left", "top", "cols", "rows", "significant_channels", "allowedlinemodes", "real_significant_bits", "sample_width", "range", "ch", "lower_limit", "higher_limit", "passcount", "detector", "dateGMT", "RefrInxMedium", "RefrInxLensMedium", "Channels", "PinholeRadius", "LambdaEx", "LambdaEm", "ExPhotonCnt", "RefInxMedium", "NumAperture", "RefInxLensMedium", "PinholeSpacing", "power", "name", "Type", "Magnification", "NA", "WorkingDistance", "Immersion", "Pinhole", "Channel .*", "Gain .*", "Shutter .*", "Position", "Size", "Port", "Cursor", "Color", "BlackLevel", "Saturation", "Gamma", "IntZoom", "Live", "Synchronize", "ShowIndex", "AutoResize", "UseUnits", "Zoom", "IgnoreAspect", "ShowCursor", "ShowAll", "Axis", "Order", "Tile", "DimViewOption", "channels", "pinholeradius", "refrinxmedium", "numaperture", "refrinxlensmedium", "image", "microscope", "stage", "filterset", "dichroic", "exc", "emm", "manufacturer", "experiment", "experimenter", "study", "metadata", "format", "icsviewer", "illumination", "exposure_time", "model", "ver", "creation", "date", "bpp", "bigendian", "size.", "physical_size.", "controller", "firmware", "pos", "position.", "mag", "na", "nm", "lamp", "built", "on", "rep", "rate", "Exposure", "Wavelength\\*"};
    private String currentIcsId;
    private String currentIdsId;
    private boolean versionTwo;
    private byte[] data;
    private long offset;
    private boolean gzip;
    private GZIPInputStream gzipStream;
    private boolean invertY;
    private boolean lifetime;
    private String labels;
    private Vector<Integer> channelLengths;
    private Vector<String> channelTypes;
    private int prevImage;
    private boolean hasInstrumentData = false;
    private boolean storedRGB = false;
    private Map<String, TrackKeyValueClass> keyValueMap = new HashMap<String, TrackKeyValueClass>();
    private int count = 0;

    public ICSReader() {
        super("Image Cytometry Standard", new String[]{"ics", "ids"});
        this.domains = new String[]{"Light Microscopy", "Fluorescence-Lifetime Imaging", "Unknown"};
        this.hasCompanionFiles = true;
    }

    public boolean isSingleFile(String id) throws FormatException, IOException {
        RandomAccessInputStream f = new RandomAccessInputStream(id);
        boolean singleFile = f.readString(17).trim().equals("ics_version\t2.0");
        f.close();
        return singleFile;
    }

    public String[] getDomains() {
        FormatTools.assertId(this.currentId, true, 1);
        String[] domain = new String[]{"Graphics"};
        if (this.getChannelDimLengths().length > 1) {
            domain[0] = "Fluorescence-Lifetime Imaging";
        } else if (this.hasInstrumentData) {
            domain[0] = "Light Microscopy";
        }
        return domain;
    }

    public int[] getChannelDimLengths() {
        FormatTools.assertId(this.currentId, true, 1);
        int[] len = new int[this.channelLengths.size()];
        for (int i = 0; i < len.length; ++i) {
            len[i] = this.channelLengths.get(i);
        }
        return len;
    }

    public String[] getChannelDimTypes() {
        FormatTools.assertId(this.currentId, true, 1);
        return this.channelTypes.toArray(new String[this.channelTypes.size()]);
    }

    public boolean isInterleaved(int subC) {
        FormatTools.assertId(this.currentId, true, 1);
        return subC == 0 && this.core[0].interleaved;
    }

    public int fileGroupOption(String id) throws FormatException, IOException {
        return 0;
    }

    public byte[] openBytes(int no, byte[] buf, int x, int y, int w, int h) throws FormatException, IOException {
        int sizeC;
        FormatTools.checkPlaneParameters(this, no, buf.length, x, y, w, h);
        int bpp = FormatTools.getBytesPerPixel(this.getPixelType());
        int len = FormatTools.getPlaneSize(this);
        int pixel = bpp * this.getRGBChannelCount();
        int rowLen = FormatTools.getPlaneSize(this, w, 1);
        int[] coordinates = this.getZCTCoords(no);
        int[] prevCoordinates = this.getZCTCoords(this.prevImage);
        if (!this.gzip) {
            this.in.seek(this.offset + (long)no * (long)len);
        } else {
            long toSkip = (long)(no - this.prevImage - 1) * (long)len;
            if (this.gzipStream == null || no <= this.prevImage) {
                FileInputStream fis = null;
                toSkip = (long)no * (long)len;
                if (this.versionTwo) {
                    fis = new FileInputStream(this.currentIcsId);
                    fis.skip(this.offset);
                } else {
                    fis = new FileInputStream(this.currentIdsId);
                    toSkip += this.offset;
                }
                try {
                    this.gzipStream = new GZIPInputStream(fis);
                }
                catch (IOException e) {
                    this.gzip = false;
                    this.in.seek(this.offset + (long)no * (long)len);
                    this.gzipStream = null;
                }
            }
            if (this.gzipStream != null) {
                while (toSkip > 0L) {
                    toSkip -= this.gzipStream.skip(toSkip);
                }
                this.data = new byte[len * (this.storedRGB ? this.getSizeC() : 1)];
                for (int toRead = this.data.length; toRead > 0; toRead -= this.gzipStream.read(this.data, this.data.length - toRead, toRead)) {
                }
            }
        }
        int n = sizeC = this.lifetime ? 1 : this.getSizeC();
        if (!this.isRGB() && sizeC > 4 && this.channelLengths.size() == 1 && this.storedRGB) {
            this.in.seek(this.offset + (long)len * (long)this.getIndex(coordinates[0], 0, coordinates[2]));
            if (!this.gzip && this.data == null) {
                this.data = new byte[len * this.getSizeC()];
                this.in.read(this.data);
            } else if (!(this.gzip || coordinates[0] == prevCoordinates[0] && coordinates[2] == prevCoordinates[2])) {
                this.in.read(this.data);
            }
            for (int row = y; row < h + y; ++row) {
                for (int col = x; col < w + x; ++col) {
                    System.arraycopy(this.data, bpp * (no % this.getSizeC() + sizeC * (row * this.getSizeX() + col)), buf, bpp * (row * w + col), bpp);
                }
            }
        } else if (this.gzip) {
            RandomAccessInputStream s = new RandomAccessInputStream(this.data);
            this.readPlane(s, x, y, w, h, buf);
            s.close();
        } else {
            this.readPlane(this.in, x, y, w, h, buf);
        }
        if (this.invertY) {
            byte[] row = new byte[rowLen];
            for (int r = 0; r < h / 2; ++r) {
                int topOffset = r * rowLen;
                int bottomOffset = (h - r - 1) * rowLen;
                System.arraycopy(buf, topOffset, row, 0, rowLen);
                System.arraycopy(buf, bottomOffset, buf, topOffset, rowLen);
                System.arraycopy(row, 0, buf, bottomOffset, rowLen);
            }
        }
        this.prevImage = no;
        return buf;
    }

    public String[] getSeriesUsedFiles(boolean noPixels) {
        String[] stringArray;
        FormatTools.assertId(this.currentId, true, 1);
        if (this.versionTwo) {
            String[] stringArray2;
            if (noPixels) {
                stringArray2 = null;
            } else {
                String[] stringArray3 = new String[1];
                stringArray2 = stringArray3;
                stringArray3[0] = this.currentIcsId;
            }
            return stringArray2;
        }
        if (noPixels) {
            String[] stringArray4 = new String[1];
            stringArray = stringArray4;
            stringArray4[0] = this.currentIcsId;
        } else {
            String[] stringArray5 = new String[2];
            stringArray5[0] = this.currentIcsId;
            stringArray = stringArray5;
            stringArray5[1] = this.currentIdsId;
        }
        return stringArray;
    }

    public void close(boolean fileOnly) throws IOException {
        super.close(fileOnly);
        if (!fileOnly) {
            this.currentIcsId = null;
            this.currentIdsId = null;
            this.data = null;
            this.versionTwo = false;
            this.gzip = false;
            this.invertY = false;
            this.lifetime = false;
            this.prevImage = 0;
            this.hasInstrumentData = false;
            this.storedRGB = false;
            if (this.gzipStream != null) {
                this.gzipStream.close();
            }
            this.gzipStream = null;
        }
    }

    protected void oldInitFile(String id) throws FormatException, IOException {
        char[] c;
        String ext;
        super.initFile(id);
        LOGGER.info("Finding companion file");
        String icsId = id;
        String idsId = id;
        int dot = id.lastIndexOf(".");
        String string = ext = dot < 0 ? "" : id.substring(dot + 1).toLowerCase();
        if (ext.equals("ics")) {
            c = idsId.toCharArray();
            int n = c.length - 2;
            c[n] = (char)(c[n] + '\u0001');
            idsId = new String(c);
        } else if (ext.equals("ids")) {
            c = icsId.toCharArray();
            int n = c.length - 2;
            c[n] = (char)(c[n] - '\u0001');
            icsId = new String(c);
        }
        if (icsId == null) {
            throw new FormatException("No ICS file found.");
        }
        Location icsFile = new Location(icsId);
        if (!icsFile.exists()) {
            throw new FormatException("ICS file not found.");
        }
        LOGGER.info("Checking file version");
        RandomAccessInputStream f = new RandomAccessInputStream(icsId);
        if (f.readString(17).trim().equals("ics_version\t2.0")) {
            this.in = new RandomAccessInputStream(icsId);
            this.versionTwo = true;
        } else {
            if (idsId == null) {
                throw new FormatException("No IDS file found.");
            }
            Location idsFile = new Location(idsId);
            if (!idsFile.exists()) {
                throw new FormatException("IDS file not found.");
            }
            this.currentIdsId = idsId;
            this.in = new RandomAccessInputStream(this.currentIdsId);
        }
        f.close();
        this.currentIcsId = icsId;
        LOGGER.info("Reading metadata");
        Double[] pixelSizes = null;
        Double[] timestamps = null;
        String[] units = null;
        String[] axes = null;
        int[] axisLengths = null;
        String byteOrder = null;
        String rFormat = null;
        String compression = null;
        RandomAccessInputStream reader = new RandomAccessInputStream(icsId);
        reader.seek(0L);
        reader.readString(NL);
        String line = reader.readString(NL);
        boolean signed = false;
        StringBuffer textBlock = new StringBuffer();
        double[] sizes = null;
        Integer[] emWaves = null;
        Integer[] exWaves = null;
        Double[] stagePos = null;
        String imageName = null;
        String date = null;
        String description = null;
        Double magnification = null;
        Double lensNA = null;
        Double workingDistance = null;
        String objectiveModel = null;
        String immersion = null;
        String lastName = null;
        Hashtable<Integer, Double> gains = new Hashtable<Integer, Double>();
        Hashtable<Integer, Double> pinholes = new Hashtable<Integer, Double>();
        Hashtable<Integer, Integer> wavelengths = new Hashtable<Integer, Integer>();
        Hashtable<Integer, String> channelNames = new Hashtable<Integer, String>();
        String laserModel = null;
        String laserManufacturer = null;
        Double laserPower = null;
        Double laserRepetitionRate = null;
        String detectorManufacturer = null;
        String detectorModel = null;
        String microscopeModel = null;
        String microscopeManufacturer = null;
        String experimentType = null;
        Double exposureTime = null;
        String filterSetModel = null;
        String dichroicModel = null;
        String excitationModel = null;
        String emissionModel = null;
        while (line != null && !line.trim().equals("end") && reader.getFilePointer() < reader.length() - 1L) {
            String[] tokens = line.split("[ \t]");
            StringBuffer key = new StringBuffer();
            for (int q = 0; q < tokens.length; ++q) {
                StringTokenizer t;
                tokens[q] = tokens[q].trim();
                if (tokens[q].length() == 0) continue;
                boolean foundValue = true;
                for (int i = 0; i < CATEGORIES.length; ++i) {
                    if (!tokens[q].matches(CATEGORIES[i])) continue;
                    foundValue = false;
                    break;
                }
                if (!foundValue) {
                    key.append(tokens[q]);
                    key.append(" ");
                    continue;
                }
                StringBuffer value = new StringBuffer(tokens[q++]);
                while (q < tokens.length) {
                    value.append(" ");
                    value.append(tokens[q].trim());
                    ++q;
                }
                String k = key.toString().trim().replaceAll("\t", " ");
                String v = value.toString().trim();
                this.addGlobalMeta(k, v);
                boolean track = true;
                Double doubleValue = null;
                try {
                    doubleValue = new Double(v);
                }
                catch (NumberFormatException e) {
                    LOGGER.debug("Could not parse double value '{}'", (Object)v, (Object)e);
                }
                if (k.equalsIgnoreCase("layout sizes")) {
                    t = new StringTokenizer(v);
                    axisLengths = new int[t.countTokens()];
                    for (int n = 0; n < axisLengths.length; ++n) {
                        try {
                            axisLengths[n] = Integer.parseInt(t.nextToken().trim());
                            continue;
                        }
                        catch (NumberFormatException e) {
                            LOGGER.debug("Could not parse axis length", (Throwable)e);
                        }
                    }
                } else if (k.equalsIgnoreCase("layout order")) {
                    t = new StringTokenizer(v);
                    axes = new String[t.countTokens()];
                    for (int n = 0; n < axes.length; ++n) {
                        axes[n] = t.nextToken().trim();
                    }
                } else if (k.equalsIgnoreCase("layout significant_bits")) {
                    this.core[0].bitsPerPixel = Integer.parseInt(v);
                } else if (k.equalsIgnoreCase("representation byte_order")) {
                    byteOrder = v;
                } else if (k.equalsIgnoreCase("representation format")) {
                    rFormat = v;
                } else if (k.equalsIgnoreCase("representation compression")) {
                    compression = v;
                } else if (k.equalsIgnoreCase("parameter scale")) {
                    pixelSizes = this.splitDoubles(v);
                } else if (k.equalsIgnoreCase("parameter t")) {
                    timestamps = this.splitDoubles(v);
                } else if (k.equalsIgnoreCase("parameter units")) {
                    units = v.split("\\s+");
                } else if (k.equalsIgnoreCase("representation sign")) {
                    signed = v.equals("signed");
                } else if (k.equalsIgnoreCase("history software") && v.indexOf("SVI") != -1) {
                    this.invertY = true;
                } else if (k.equalsIgnoreCase("filename")) {
                    imageName = v;
                } else if (k.equalsIgnoreCase("history date") || k.equalsIgnoreCase("history created on")) {
                    if (v.indexOf(" ") == -1) continue;
                    date = v.substring(0, v.lastIndexOf(" "));
                    date = DateTools.formatDate(date, DATE_FORMATS);
                } else if (k.equalsIgnoreCase("history creation date")) {
                    date = DateTools.formatDate(v, DATE_FORMATS);
                } else if (k.equalsIgnoreCase("history type")) {
                    if (v.equalsIgnoreCase("time resolved") || v.equalsIgnoreCase("FluorescenceLifetime")) {
                        this.lifetime = true;
                    }
                    experimentType = v;
                } else if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM) {
                    Integer n;
                    String[] waves;
                    if (k.equalsIgnoreCase("sensor s_params LambdaEm")) {
                        waves = v.split(" ");
                        emWaves = new Integer[waves.length];
                        for (int n2 = 0; n2 < emWaves.length; ++n2) {
                            try {
                                emWaves[n2] = new Integer((int)Double.parseDouble(waves[n2]));
                                continue;
                            }
                            catch (NumberFormatException e) {
                                LOGGER.debug("Could not parse emission wavelength", (Throwable)e);
                            }
                        }
                    } else if (k.equalsIgnoreCase("sensor s_params LambdaEx")) {
                        waves = v.split(" ");
                        exWaves = new Integer[waves.length];
                        for (int n3 = 0; n3 < exWaves.length; ++n3) {
                            try {
                                exWaves[n3] = new Integer((int)Double.parseDouble(waves[n3]));
                                continue;
                            }
                            catch (NumberFormatException e) {
                                LOGGER.debug("Could not parse excitation wavelength", (Throwable)e);
                            }
                        }
                    } else if (k.equalsIgnoreCase("history") || k.equalsIgnoreCase("history text")) {
                        textBlock.append(v);
                        textBlock.append("\n");
                        this.metadata.remove(k);
                        track = false;
                    } else if (k.startsWith("history gain")) {
                        n = new Integer(0);
                        try {
                            n = new Integer(k.substring(12).trim());
                            n = new Integer(n - 1);
                        }
                        catch (NumberFormatException e) {
                            // empty catch block
                        }
                        if (doubleValue != null) {
                            gains.put(n, doubleValue);
                        }
                    } else if (k.startsWith("history laser") && k.endsWith("wavelength")) {
                        int laser = Integer.parseInt(k.substring(13, k.indexOf(" ", 13))) - 1;
                        v = v.replaceAll("nm", "").trim();
                        try {
                            wavelengths.put(new Integer(laser), new Integer(v));
                        }
                        catch (NumberFormatException e) {
                            LOGGER.debug("Could not parse wavelength", (Throwable)e);
                        }
                    } else if (k.equalsIgnoreCase("history Wavelength*")) {
                        waves = v.split(" ");
                        for (int i = 0; i < waves.length; ++i) {
                            wavelengths.put(new Integer(i), new Integer(waves[i]));
                        }
                    } else if (k.equalsIgnoreCase("history laser manufacturer")) {
                        laserManufacturer = v;
                    } else if (k.equalsIgnoreCase("history laser model")) {
                        laserModel = v;
                    } else if (k.equalsIgnoreCase("history laser power")) {
                        try {
                            laserPower = new Double(v);
                        }
                        catch (NumberFormatException e) {}
                    } else if (k.equalsIgnoreCase("history laser rep rate")) {
                        String repRate = v;
                        if (repRate.indexOf(" ") != -1) {
                            repRate = repRate.substring(0, repRate.lastIndexOf(" "));
                        }
                        laserRepetitionRate = new Double(repRate);
                    } else if (k.equalsIgnoreCase("history objective type") || k.equalsIgnoreCase("history objective")) {
                        objectiveModel = v;
                    } else if (k.equalsIgnoreCase("history objective immersion")) {
                        immersion = v;
                    } else if (k.equalsIgnoreCase("history objective NA")) {
                        lensNA = doubleValue;
                    } else if (k.equalsIgnoreCase("history objective WorkingDistance")) {
                        workingDistance = doubleValue;
                    } else if (k.equalsIgnoreCase("history objective magnification") || k.equalsIgnoreCase("history objective mag")) {
                        magnification = doubleValue;
                    } else if (k.equalsIgnoreCase("history camera manufacturer")) {
                        detectorManufacturer = v;
                    } else if (k.equalsIgnoreCase("history camera model")) {
                        detectorModel = v;
                    } else if (k.equalsIgnoreCase("sensor s_params PinholeRadius")) {
                        String[] pins = v.split(" ");
                        int channel = 0;
                        for (int n4 = 0; n4 < pins.length; ++n4) {
                            if (pins[n4].trim().equals("")) continue;
                            try {
                                pinholes.put(new Integer(channel++), new Double(pins[n4]));
                                continue;
                            }
                            catch (NumberFormatException e) {
                                LOGGER.debug("Could not parse pinhole", (Throwable)e);
                            }
                        }
                    } else if (k.equalsIgnoreCase("history author") || k.equalsIgnoreCase("history experimenter")) {
                        lastName = v;
                    } else if (k.equalsIgnoreCase("history extents")) {
                        String[] lengths = v.split(" ");
                        sizes = new double[lengths.length];
                        for (int n5 = 0; n5 < sizes.length; ++n5) {
                            try {
                                sizes[n5] = Double.parseDouble(lengths[n5].trim());
                                continue;
                            }
                            catch (NumberFormatException e) {
                                LOGGER.debug("Could not parse axis length", (Throwable)e);
                            }
                        }
                    } else if (k.equalsIgnoreCase("history stage_xyzum")) {
                        String[] positions = v.split(" ");
                        stagePos = new Double[positions.length];
                        for (int n6 = 0; n6 < stagePos.length; ++n6) {
                            try {
                                stagePos[n6] = new Double(positions[n6]);
                                continue;
                            }
                            catch (NumberFormatException e) {
                                LOGGER.debug("Could not parse stage position", (Throwable)e);
                            }
                        }
                    } else if (k.equalsIgnoreCase("history stage positionx")) {
                        if (stagePos == null) {
                            stagePos = new Double[3];
                        }
                        stagePos[0] = new Double(v);
                    } else if (k.equalsIgnoreCase("history stage positiony")) {
                        if (stagePos == null) {
                            stagePos = new Double[3];
                        }
                        stagePos[1] = new Double(v);
                    } else if (k.equalsIgnoreCase("history stage positionz")) {
                        if (stagePos == null) {
                            stagePos = new Double[3];
                        }
                        stagePos[2] = new Double(v);
                    } else if (k.equalsIgnoreCase("history other text")) {
                        description = v;
                    } else if (k.startsWith("history step") && k.endsWith("name")) {
                        n = new Integer(k.substring(12, k.indexOf(" ", 12)));
                        channelNames.put(n, v);
                    } else if (k.equalsIgnoreCase("parameter ch")) {
                        String[] names = v.split(" ");
                        for (int n7 = 0; n7 < names.length; ++n7) {
                            channelNames.put(new Integer(n7), names[n7].trim());
                        }
                    } else if (k.equalsIgnoreCase("history cube")) {
                        channelNames.put(new Integer(channelNames.size()), v);
                    } else if (k.equalsIgnoreCase("history cube emm nm")) {
                        if (emWaves == null) {
                            emWaves = new Integer[]{new Integer(v.split(" ")[1].trim())};
                        }
                    } else if (k.equalsIgnoreCase("history cube exc nm")) {
                        if (exWaves == null) {
                            exWaves = new Integer[]{new Integer(v.split(" ")[1].trim())};
                        }
                    } else if (k.equalsIgnoreCase("history microscope")) {
                        microscopeModel = v;
                    } else if (k.equalsIgnoreCase("history manufacturer")) {
                        microscopeManufacturer = v;
                    } else if (k.equalsIgnoreCase("history Exposure")) {
                        String expTime = v;
                        if (expTime.indexOf(" ") != -1) {
                            expTime = expTime.substring(0, expTime.indexOf(" "));
                        }
                        exposureTime = new Double(expTime);
                    } else if (k.equalsIgnoreCase("history filterset")) {
                        filterSetModel = v;
                    } else if (k.equalsIgnoreCase("history filterset dichroic name")) {
                        dichroicModel = v;
                    } else if (k.equalsIgnoreCase("history filterset exc name")) {
                        excitationModel = v;
                    } else if (k.equalsIgnoreCase("history filterset emm name")) {
                        emissionModel = v;
                    }
                }
                if (!track) continue;
                this.trackKeyValues(id, k, v, "");
            }
            line = reader.readString(NL);
        }
        reader.close();
        this.hasInstrumentData = emWaves != null || exWaves != null || lensNA != null || stagePos != null || magnification != null || workingDistance != null || objectiveModel != null || immersion != null;
        this.addGlobalMeta("history text", textBlock.toString());
        LOGGER.info("Populating core metadata");
        this.core[0].rgb = false;
        this.core[0].dimensionOrder = "XY";
        this.channelLengths = new Vector();
        this.channelTypes = new Vector();
        int bitsPerPixel = 0;
        for (int i = 0; i < axes.length && i < axisLengths.length; ++i) {
            if (axes[i].equals("bits")) {
                bitsPerPixel = axisLengths[i];
                while (bitsPerPixel % 8 != 0) {
                    ++bitsPerPixel;
                }
                if (bitsPerPixel != 24 && bitsPerPixel != 48) continue;
                bitsPerPixel /= 3;
                continue;
            }
            if (axes[i].equals("x")) {
                this.core[0].sizeX = axisLengths[i];
                continue;
            }
            if (axes[i].equals("y")) {
                this.core[0].sizeY = axisLengths[i];
                continue;
            }
            if (axes[i].equals("z")) {
                this.core[0].sizeZ = axisLengths[i];
                if (this.getDimensionOrder().indexOf("Z") != -1) continue;
                this.core[0].dimensionOrder = this.core[0].dimensionOrder + "Z";
                continue;
            }
            if (axes[i].equals("t")) {
                this.core[0].sizeT = this.getSizeT() == 0 ? axisLengths[i] : (this.core[0].sizeT *= axisLengths[i]);
                if (this.getDimensionOrder().indexOf("T") != -1) continue;
                this.core[0].dimensionOrder = this.core[0].dimensionOrder + "T";
                continue;
            }
            this.core[0].sizeC = this.core[0].sizeC == 0 ? axisLengths[i] : (this.core[0].sizeC *= axisLengths[i]);
            this.channelLengths.add(new Integer(axisLengths[i]));
            this.storedRGB = this.getSizeX() == 0;
            boolean bl = this.core[0].rgb = this.getSizeX() == 0 && this.getSizeC() <= 4 && this.getSizeC() > 1;
            if (this.getDimensionOrder().indexOf("C") == -1) {
                this.core[0].dimensionOrder = this.core[0].dimensionOrder + "C";
            }
            if (axes[i].startsWith("c")) {
                this.channelTypes.add("Channel");
                continue;
            }
            if (axes[i].equals("p")) {
                this.channelTypes.add("Phase");
                continue;
            }
            if (axes[i].equals("f")) {
                this.channelTypes.add("Frequency");
                continue;
            }
            this.channelTypes.add("");
        }
        if (this.channelLengths.size() == 0) {
            this.channelLengths.add(new Integer(1));
            this.channelTypes.add("Channel");
        }
        this.core[0].dimensionOrder = MetadataTools.makeSaneDimensionOrder(this.getDimensionOrder());
        if (this.getSizeZ() == 0) {
            this.core[0].sizeZ = 1;
        }
        if (this.getSizeC() == 0) {
            this.core[0].sizeC = 1;
        }
        if (this.getSizeT() == 0) {
            this.core[0].sizeT = 1;
        }
        this.core[0].interleaved = this.isRGB();
        this.core[0].imageCount = this.getSizeZ() * this.getSizeT();
        if (!this.isRGB()) {
            this.core[0].imageCount *= this.getSizeC();
        }
        this.core[0].indexed = false;
        this.core[0].falseColor = false;
        this.core[0].metadataComplete = true;
        this.core[0].littleEndian = true;
        if (this.lifetime) {
            int binCount = this.core[0].sizeZ;
            this.core[0].sizeZ = this.core[0].sizeC;
            this.core[0].sizeC = binCount;
            this.core[0].cLengths = new int[]{binCount};
            this.core[0].cTypes = new String[]{"Lifetime"};
            String newOrder = this.core[0].dimensionOrder.replace("Z", "x");
            newOrder = newOrder.replace("C", "Z");
            this.core[0].dimensionOrder = newOrder = newOrder.replace("x", "C");
        }
        if (byteOrder != null) {
            String firstByte = byteOrder.split(" ")[0];
            int first = Integer.parseInt(firstByte);
            this.core[0].littleEndian = rFormat.equals("real") ? first == 1 : first != 1;
        }
        boolean bl = this.gzip = compression == null ? false : compression.equals("gzip");
        if (this.versionTwo) {
            String s = this.in.readString(NL);
            while (!s.trim().equals("end")) {
                s = this.in.readString(NL);
            }
        }
        this.offset = this.in.getFilePointer();
        int bytes = bitsPerPixel / 8;
        if (bitsPerPixel < 32) {
            this.core[0].littleEndian = !this.isLittleEndian();
        }
        boolean fp = rFormat.equals("real");
        this.core[0].pixelType = FormatTools.pixelTypeFromBytes(bytes, signed, fp);
        LOGGER.info("Populating OME metadata");
        MetadataStore store = this.makeFilterMetadata();
        MetadataTools.populatePixels(store, this, true);
        store.setImageName(imageName, 0);
        if (date != null) {
            store.setImageAcquiredDate(date, 0);
        } else {
            MetadataTools.setDefaultCreationDate(store, id, 0);
        }
        if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM) {
            store.setImageDescription(description, 0);
            String instrumentID = MetadataTools.createLSID("Instrument", 0);
            store.setInstrumentID(instrumentID, 0);
            store.setMicroscopeModel(microscopeModel, 0);
            store.setMicroscopeManufacturer(microscopeManufacturer, 0);
            store.setImageInstrumentRef(instrumentID, 0);
            store.setExperimentID(MetadataTools.createLSID("Experiment", 0), 0);
            store.setExperimentType(this.getExperimentType(experimentType), 0);
            if (pixelSizes != null) {
                for (int i = 0; i < pixelSizes.length; ++i) {
                    String unit;
                    Double pixelSize = pixelSizes[i];
                    String axis = axes != null && axes.length > i ? axes[i] : "";
                    String string2 = unit = units != null && units.length > i ? units[i] : "";
                    if (axis.equals("x")) {
                        if (!(pixelSize > 0.0) || !this.checkUnit(unit, "um", "microns")) continue;
                        store.setPixelsPhysicalSizeX(new PositiveFloat(pixelSize), 0);
                        continue;
                    }
                    if (axis.equals("y")) {
                        if (!(pixelSize > 0.0) || !this.checkUnit(unit, "um", "microns")) continue;
                        store.setPixelsPhysicalSizeY(new PositiveFloat(pixelSize), 0);
                        continue;
                    }
                    if (axis.equals("z")) {
                        if (!(pixelSize > 0.0) || !this.checkUnit(unit, "um", "microns")) continue;
                        store.setPixelsPhysicalSizeZ(new PositiveFloat(pixelSize), 0);
                        continue;
                    }
                    if (!axis.equals("t") || !this.checkUnit(unit, "ms")) continue;
                    store.setPixelsTimeIncrement(1000.0 * pixelSize, 0);
                }
            } else if (sizes != null) {
                if (sizes.length > 0 && sizes[0] > 0.0) {
                    store.setPixelsPhysicalSizeX(new PositiveFloat(Double.valueOf((double)sizes[0])), 0);
                }
                if (sizes.length > 1) {
                    sizes[1] = sizes[1] / (double)this.getSizeY();
                    if (sizes[1] > 0.0) {
                        store.setPixelsPhysicalSizeY(new PositiveFloat(Double.valueOf((double)sizes[1])), 0);
                    }
                }
            }
            if (timestamps != null) {
                for (int t = 0; t < timestamps.length && t < this.getSizeT(); ++t) {
                    double deltaT;
                    if (timestamps[t] == null || Double.isNaN(deltaT = timestamps[t].doubleValue())) continue;
                    for (int z = 0; z < this.getSizeZ(); ++z) {
                        for (int c2 = 0; c2 < this.getEffectiveSizeC(); ++c2) {
                            int index = this.getIndex(z, c2, t);
                            store.setPlaneDeltaT(deltaT, 0, index);
                        }
                    }
                }
            }
            for (int i = 0; i < this.getEffectiveSizeC(); ++i) {
                if (channelNames.containsKey(i)) {
                    store.setChannelName((String)channelNames.get(i), 0, i);
                }
                if (pinholes.containsKey(i)) {
                    store.setChannelPinholeSize((Double)pinholes.get(i), 0, i);
                }
                if (emWaves != null && i < emWaves.length && emWaves[i] > 0) {
                    store.setChannelEmissionWavelength(new PositiveInteger(emWaves[i]), 0, i);
                }
                if (exWaves == null || i >= exWaves.length || exWaves[i] <= 0) continue;
                store.setChannelExcitationWavelength(new PositiveInteger(exWaves[i]), 0, i);
            }
            Object[] lasers = wavelengths.keySet().toArray(new Integer[0]);
            Arrays.sort(lasers);
            for (int i = 0; i < lasers.length; ++i) {
                store.setLaserID(MetadataTools.createLSID("LightSource", 0, i), 0, i);
                store.setLaserWavelength(new PositiveInteger((Integer)wavelengths.get(lasers[i])), 0, i);
                store.setLaserType(this.getLaserType("Other"), 0, i);
                store.setLaserLaserMedium(this.getLaserMedium("Other"), 0, i);
                store.setLaserManufacturer(laserManufacturer, 0, i);
                store.setLaserModel(laserModel, 0, i);
                store.setLaserPower(laserPower, 0, i);
                store.setLaserRepetitionRate(laserRepetitionRate, 0, i);
            }
            if (lasers.length == 0 && laserManufacturer != null) {
                store.setLaserID(MetadataTools.createLSID("LightSource", 0, 0), 0, 0);
                store.setLaserType(this.getLaserType("Other"), 0, 0);
                store.setLaserLaserMedium(this.getLaserMedium("Other"), 0, 0);
                store.setLaserManufacturer(laserManufacturer, 0, 0);
                store.setLaserModel(laserModel, 0, 0);
                store.setLaserPower(laserPower, 0, 0);
                store.setLaserRepetitionRate(laserRepetitionRate, 0, 0);
            }
            if (filterSetModel != null) {
                store.setFilterSetID(MetadataTools.createLSID("FilterSet", 0, 0), 0, 0);
                store.setFilterSetModel(filterSetModel, 0, 0);
                String dichroicID = MetadataTools.createLSID("Dichroic", 0, 0);
                String emFilterID = MetadataTools.createLSID("Filter", 0, 0);
                String exFilterID = MetadataTools.createLSID("Filter", 0, 1);
                store.setDichroicID(dichroicID, 0, 0);
                store.setDichroicModel(dichroicModel, 0, 0);
                store.setFilterSetDichroicRef(dichroicID, 0, 0);
                store.setFilterID(emFilterID, 0, 0);
                store.setFilterModel(emissionModel, 0, 0);
                store.setFilterSetEmissionFilterRef(emFilterID, 0, 0, 0);
                store.setFilterID(exFilterID, 0, 1);
                store.setFilterModel(excitationModel, 0, 1);
                store.setFilterSetExcitationFilterRef(exFilterID, 0, 0, 0);
            }
            if (objectiveModel != null) {
                store.setObjectiveModel(objectiveModel, 0, 0);
            }
            if (immersion == null) {
                immersion = "Other";
            }
            store.setObjectiveImmersion(this.getImmersion(immersion), 0, 0);
            if (lensNA != null) {
                store.setObjectiveLensNA(lensNA, 0, 0);
            }
            if (workingDistance != null) {
                store.setObjectiveWorkingDistance(workingDistance, 0, 0);
            }
            if (magnification != null) {
                store.setObjectiveCalibratedMagnification(magnification, 0, 0);
            }
            store.setObjectiveCorrection(this.getCorrection("Other"), 0, 0);
            String objectiveID = MetadataTools.createLSID("Objective", 0, 0);
            store.setObjectiveID(objectiveID, 0, 0);
            store.setImageObjectiveSettingsID(objectiveID, 0);
            String detectorID = MetadataTools.createLSID("Detector", 0, 0);
            store.setDetectorID(detectorID, 0, 0);
            store.setDetectorManufacturer(detectorManufacturer, 0, 0);
            store.setDetectorModel(detectorModel, 0, 0);
            store.setDetectorType(this.getDetectorType("Other"), 0, 0);
            for (Integer key : gains.keySet()) {
                int index = key;
                if (index >= this.getEffectiveSizeC()) continue;
                store.setDetectorSettingsGain((Double)gains.get(key), 0, index);
                store.setDetectorSettingsID(detectorID, 0, index);
            }
            if (lastName != null) {
                String experimenterID = MetadataTools.createLSID("Experimenter", 0);
                store.setExperimenterID(experimenterID, 0);
                store.setExperimenterLastName(lastName, 0);
                store.setExperimenterDisplayName(lastName, 0);
            }
            if (stagePos != null) {
                for (int i = 0; i < this.getImageCount(); ++i) {
                    if (stagePos.length > 0) {
                        store.setPlanePositionX(stagePos[0], 0, i);
                        this.addGlobalMeta("X position for position #1", stagePos[0]);
                    }
                    if (stagePos.length > 1) {
                        store.setPlanePositionY(stagePos[1], 0, i);
                        this.addGlobalMeta("Y position for position #1", stagePos[1]);
                    }
                    if (stagePos.length <= 2) continue;
                    store.setPlanePositionZ(stagePos[2], 0, i);
                    this.addGlobalMeta("Z position for position #1", stagePos[2]);
                }
            }
            if (exposureTime != null) {
                for (int i = 0; i < this.getImageCount(); ++i) {
                    store.setPlaneExposureTime(exposureTime, 0, i);
                }
            }
        }
    }

    protected void initFile(String id) throws FormatException, IOException {
        char[] c;
        String ext;
        super.initFile(id);
        LOGGER.info("Finding companion file");
        String icsId = id;
        String idsId = id;
        int dot = id.lastIndexOf(".");
        String string = ext = dot < 0 ? "" : id.substring(dot + 1).toLowerCase();
        if (ext.equals("ics")) {
            c = idsId.toCharArray();
            int n = c.length - 2;
            c[n] = (char)(c[n] + '\u0001');
            idsId = new String(c);
        } else if (ext.equals("ids")) {
            c = icsId.toCharArray();
            int n = c.length - 2;
            c[n] = (char)(c[n] - '\u0001');
            icsId = new String(c);
        }
        if (icsId == null) {
            throw new FormatException("No ICS file found.");
        }
        Location icsFile = new Location(icsId);
        if (!icsFile.exists()) {
            throw new FormatException("ICS file not found.");
        }
        LOGGER.info("Checking file version");
        RandomAccessInputStream f = new RandomAccessInputStream(icsId);
        if (f.readString(17).trim().equals("ics_version\t2.0")) {
            this.in = new RandomAccessInputStream(icsId);
            this.versionTwo = true;
        } else {
            if (idsId == null) {
                throw new FormatException("No IDS file found.");
            }
            Location idsFile = new Location(idsId);
            if (!idsFile.exists()) {
                throw new FormatException("IDS file not found.");
            }
            this.currentIdsId = idsId;
            this.in = new RandomAccessInputStream(this.currentIdsId);
        }
        f.close();
        this.currentIcsId = icsId;
        LOGGER.info("Reading metadata");
        Double[] pixelSizes = null;
        Double[] timestamps = null;
        String[] units = null;
        String[] axes = null;
        int[] axisLengths = null;
        String byteOrder = null;
        String rFormat = null;
        String compression = null;
        RandomAccessInputStream reader = new RandomAccessInputStream(icsId);
        reader.seek(0L);
        reader.readString(NL);
        String line = reader.readString(NL);
        boolean signed = false;
        StringBuffer textBlock = new StringBuffer();
        double[] sizes = null;
        Integer[] emWaves = null;
        Integer[] exWaves = null;
        Double[] stagePos = null;
        String imageName = null;
        String date = null;
        String description = null;
        Double magnification = null;
        Double lensNA = null;
        Double workingDistance = null;
        String objectiveModel = null;
        String immersion = null;
        String lastName = null;
        Hashtable<Integer, Double> gains = new Hashtable<Integer, Double>();
        Hashtable<Integer, Double> pinholes = new Hashtable<Integer, Double>();
        Hashtable<Integer, Integer> wavelengths = new Hashtable<Integer, Integer>();
        Hashtable<Integer, String> channelNames = new Hashtable<Integer, String>();
        String laserModel = null;
        String laserManufacturer = null;
        Double laserPower = null;
        Double laserRepetitionRate = null;
        String detectorManufacturer = null;
        String detectorModel = null;
        String microscopeModel = null;
        String microscopeManufacturer = null;
        String experimentType = null;
        Double exposureTime = null;
        String filterSetModel = null;
        String dichroicModel = null;
        String excitationModel = null;
        String emissionModel = null;
        while (line != null && !line.trim().equals("end") && reader.getFilePointer() < reader.length() - 1L) {
            if ((line = line.trim()).length() > 0) {
                String value;
                String key;
                String[] tokens = this.tokenize(line);
                String token0 = tokens[0].toLowerCase();
                String[] keyValue = null;
                if (token0.equals("ics_version")) {
                    String value2 = this.concatenateTokens(tokens, 1, tokens.length);
                    this.addGlobalMeta(token0, value2);
                    this.trackKeyValues(id, token0, value2, "");
                } else if (token0.equals("filename")) {
                    imageName = this.concatenateTokens(tokens, 1, tokens.length);
                    this.addGlobalMeta(token0, imageName);
                    this.trackKeyValues(id, token0, imageName, "");
                } else if (token0.equals("layout")) {
                    keyValue = this.findKeyValue(tokens, LAYOUT_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    this.addGlobalMeta(key, value);
                    this.trackKeyValues(id, key, value, "");
                    if (key.equalsIgnoreCase("layout sizes")) {
                        StringTokenizer t = new StringTokenizer(value);
                        axisLengths = new int[t.countTokens()];
                        for (int n = 0; n < axisLengths.length; ++n) {
                            try {
                                axisLengths[n] = Integer.parseInt(t.nextToken().trim());
                                continue;
                            }
                            catch (NumberFormatException e) {
                                LOGGER.debug("Could not parse axis length", (Throwable)e);
                            }
                        }
                    } else if (key.equalsIgnoreCase("layout order")) {
                        StringTokenizer t = new StringTokenizer(value);
                        axes = new String[t.countTokens()];
                        for (int n = 0; n < axes.length; ++n) {
                            axes[n] = t.nextToken().trim();
                        }
                    } else if (key.equalsIgnoreCase("layout significant_bits")) {
                        this.core[0].bitsPerPixel = Integer.parseInt(value);
                    }
                } else if (token0.equals("representation")) {
                    keyValue = this.findKeyValue(tokens, REPRESENTATION_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    this.addGlobalMeta(key, value);
                    this.trackKeyValues(id, key, value, "");
                    if (key.equalsIgnoreCase("representation byte_order")) {
                        byteOrder = value;
                    } else if (key.equalsIgnoreCase("representation format")) {
                        rFormat = value;
                    } else if (key.equalsIgnoreCase("representation compression")) {
                        compression = value;
                    } else if (key.equalsIgnoreCase("representation sign")) {
                        signed = value.equals("signed");
                    }
                } else if (token0.equals("parameter")) {
                    keyValue = this.findKeyValue(tokens, PARAMETER_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    this.addGlobalMeta(key, value);
                    this.trackKeyValues(id, key, value, "");
                    if (key.equalsIgnoreCase("parameter scale")) {
                        pixelSizes = this.splitDoubles(value);
                    } else if (key.equalsIgnoreCase("parameter t")) {
                        timestamps = this.splitDoubles(value);
                    } else if (key.equalsIgnoreCase("parameter units")) {
                        units = value.split("\\s+");
                    }
                    if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM && key.equalsIgnoreCase("parameter ch")) {
                        String[] names = value.split(" ");
                        for (int n = 0; n < names.length; ++n) {
                            channelNames.put(new Integer(n), names[n].trim());
                        }
                    }
                } else if (token0.equals("history")) {
                    keyValue = this.findKeyValue(tokens, HISTORY_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    this.addGlobalMeta(key, value);
                    boolean track = true;
                    Double doubleValue = null;
                    try {
                        doubleValue = new Double(value);
                    }
                    catch (NumberFormatException e) {
                        LOGGER.debug("Could not parse double value '{}'", (Object)value, (Object)e);
                    }
                    if (key.equalsIgnoreCase("history software") && value.indexOf("SVI") != -1) {
                        this.invertY = true;
                    } else if (key.equalsIgnoreCase("history date") || key.equalsIgnoreCase("history created on")) {
                        if (value.indexOf(" ") > 0) {
                            date = value.substring(0, value.lastIndexOf(" "));
                            date = DateTools.formatDate(date, DATE_FORMATS);
                        }
                    } else if (key.equalsIgnoreCase("history creation date")) {
                        date = DateTools.formatDate(value, DATE_FORMATS);
                    } else if (key.equalsIgnoreCase("history type")) {
                        if (value.equalsIgnoreCase("time resolved") || value.equalsIgnoreCase("FluorescenceLifetime")) {
                            this.lifetime = true;
                        }
                        experimentType = value;
                    } else if (key.equalsIgnoreCase("history labels")) {
                        this.labels = value;
                    } else if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM) {
                        if (key.equalsIgnoreCase("history") || key.equalsIgnoreCase("history text")) {
                            textBlock.append(value);
                            textBlock.append("\n");
                            this.metadata.remove(key);
                            track = false;
                        } else if (key.startsWith("history gain")) {
                            Integer n = new Integer(0);
                            try {
                                n = new Integer(key.substring(12).trim());
                                n = new Integer(n - 1);
                            }
                            catch (NumberFormatException e) {
                                // empty catch block
                            }
                            if (doubleValue != null) {
                                gains.put(n, doubleValue);
                            }
                        } else if (key.startsWith("history laser") && key.endsWith("wavelength")) {
                            int laser = Integer.parseInt(key.substring(13, key.indexOf(" ", 13))) - 1;
                            value = value.replaceAll("nm", "").trim();
                            try {
                                wavelengths.put(new Integer(laser), new Integer(value));
                            }
                            catch (NumberFormatException e) {
                                LOGGER.debug("Could not parse wavelength", (Throwable)e);
                            }
                        } else if (key.equalsIgnoreCase("history Wavelength*")) {
                            String[] waves = value.split(" ");
                            for (int i = 0; i < waves.length; ++i) {
                                wavelengths.put(new Integer(i), new Integer(waves[i]));
                            }
                        } else if (key.equalsIgnoreCase("history laser manufacturer")) {
                            laserManufacturer = value;
                        } else if (key.equalsIgnoreCase("history laser model")) {
                            laserModel = value;
                        } else if (key.equalsIgnoreCase("history laser power")) {
                            try {
                                laserPower = new Double(value);
                            }
                            catch (NumberFormatException e) {}
                        } else if (key.equalsIgnoreCase("history laser rep rate")) {
                            String repRate = value;
                            if (repRate.indexOf(" ") != -1) {
                                repRate = repRate.substring(0, repRate.lastIndexOf(" "));
                            }
                            laserRepetitionRate = new Double(repRate);
                        } else if (key.equalsIgnoreCase("history objective type") || key.equalsIgnoreCase("history objective")) {
                            objectiveModel = value;
                        } else if (key.equalsIgnoreCase("history objective immersion")) {
                            immersion = value;
                        } else if (key.equalsIgnoreCase("history objective NA")) {
                            lensNA = doubleValue;
                        } else if (key.equalsIgnoreCase("history objective WorkingDistance")) {
                            workingDistance = doubleValue;
                        } else if (key.equalsIgnoreCase("history objective magnification") || key.equalsIgnoreCase("history objective mag")) {
                            magnification = doubleValue;
                        } else if (key.equalsIgnoreCase("history camera manufacturer")) {
                            detectorManufacturer = value;
                        } else if (key.equalsIgnoreCase("history camera model")) {
                            detectorModel = value;
                        } else if (key.equalsIgnoreCase("history author") || key.equalsIgnoreCase("history experimenter")) {
                            lastName = value;
                        } else if (key.equalsIgnoreCase("history extents")) {
                            String[] lengths = value.split(" ");
                            sizes = new double[lengths.length];
                            for (int n = 0; n < sizes.length; ++n) {
                                try {
                                    sizes[n] = Double.parseDouble(lengths[n].trim());
                                    continue;
                                }
                                catch (NumberFormatException e) {
                                    LOGGER.debug("Could not parse axis length", (Throwable)e);
                                }
                            }
                        } else if (key.equalsIgnoreCase("history stage_xyzum")) {
                            String[] positions = value.split(" ");
                            stagePos = new Double[positions.length];
                            for (int n = 0; n < stagePos.length; ++n) {
                                try {
                                    stagePos[n] = new Double(positions[n]);
                                    continue;
                                }
                                catch (NumberFormatException e) {
                                    LOGGER.debug("Could not parse stage position", (Throwable)e);
                                }
                            }
                        } else if (key.equalsIgnoreCase("history stage positionx")) {
                            if (stagePos == null) {
                                stagePos = new Double[3];
                            }
                            stagePos[0] = new Double(value);
                        } else if (key.equalsIgnoreCase("history stage positiony")) {
                            if (stagePos == null) {
                                stagePos = new Double[3];
                            }
                            stagePos[1] = new Double(value);
                        } else if (key.equalsIgnoreCase("history stage positionz")) {
                            if (stagePos == null) {
                                stagePos = new Double[3];
                            }
                            stagePos[2] = new Double(value);
                        } else if (key.equalsIgnoreCase("history other text")) {
                            description = value;
                        } else if (key.startsWith("history step") && key.endsWith("name")) {
                            Integer n = new Integer(key.substring(12, key.indexOf(" ", 12)));
                            channelNames.put(n, value);
                        } else if (key.equalsIgnoreCase("history cube")) {
                            channelNames.put(new Integer(channelNames.size()), value);
                        } else if (key.equalsIgnoreCase("history cube emm nm")) {
                            if (emWaves == null) {
                                emWaves = new Integer[]{new Integer(value.split(" ")[1].trim())};
                            }
                        } else if (key.equalsIgnoreCase("history cube exc nm")) {
                            if (exWaves == null) {
                                exWaves = new Integer[]{new Integer(value.split(" ")[1].trim())};
                            }
                        } else if (key.equalsIgnoreCase("history microscope")) {
                            microscopeModel = value;
                        } else if (key.equalsIgnoreCase("history manufacturer")) {
                            microscopeManufacturer = value;
                        } else if (key.equalsIgnoreCase("history Exposure")) {
                            String expTime = value;
                            if (expTime.indexOf(" ") != -1) {
                                expTime = expTime.substring(0, expTime.indexOf(" "));
                            }
                            exposureTime = new Double(expTime);
                        } else if (key.equalsIgnoreCase("history filterset")) {
                            filterSetModel = value;
                        } else if (key.equalsIgnoreCase("history filterset dichroic name")) {
                            dichroicModel = value;
                        } else if (key.equalsIgnoreCase("history filterset exc name")) {
                            excitationModel = value;
                        } else if (key.equalsIgnoreCase("history filterset emm name")) {
                            emissionModel = value;
                        }
                    }
                    if (track) {
                        this.trackKeyValues(id, key, value, "");
                    }
                } else if (token0.equals("document")) {
                    keyValue = this.findKeyValue(tokens, DOCUMENT_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    this.addGlobalMeta(key, value);
                    this.trackKeyValues(id, key, value, "");
                } else if (token0.equals("sensor")) {
                    keyValue = this.findKeyValue(tokens, SENSOR_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    this.addGlobalMeta(key, value);
                    this.trackKeyValues(id, key, value, "");
                    if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM) {
                        if (key.equalsIgnoreCase("sensor s_params LambdaEm")) {
                            String[] waves = value.split(" ");
                            emWaves = new Integer[waves.length];
                            for (int n = 0; n < emWaves.length; ++n) {
                                try {
                                    emWaves[n] = new Integer((int)Double.parseDouble(waves[n]));
                                    continue;
                                }
                                catch (NumberFormatException e) {
                                    LOGGER.debug("Could not parse emission wavelength", (Throwable)e);
                                }
                            }
                        } else if (key.equalsIgnoreCase("sensor s_params LambdaEx")) {
                            String[] waves = value.split(" ");
                            exWaves = new Integer[waves.length];
                            for (int n = 0; n < exWaves.length; ++n) {
                                try {
                                    exWaves[n] = new Integer((int)Double.parseDouble(waves[n]));
                                    continue;
                                }
                                catch (NumberFormatException e) {
                                    LOGGER.debug("Could not parse excitation wavelength", (Throwable)e);
                                }
                            }
                        } else if (key.equalsIgnoreCase("sensor s_params PinholeRadius")) {
                            String[] pins = value.split(" ");
                            int channel = 0;
                            for (int n = 0; n < pins.length; ++n) {
                                if (pins[n].trim().equals("")) continue;
                                try {
                                    pinholes.put(new Integer(channel++), new Double(pins[n]));
                                    continue;
                                }
                                catch (NumberFormatException e) {
                                    LOGGER.debug("Could not parse pinhole", (Throwable)e);
                                }
                            }
                        }
                    }
                } else if (token0.equals("view")) {
                    keyValue = this.findKeyValue(tokens, VIEW_KEYS);
                    key = keyValue[0];
                    value = keyValue[1];
                    if (key.equalsIgnoreCase("view view color lib lut")) {
                        int redIndex = value.toLowerCase().lastIndexOf("red");
                        int greenIndex = value.toLowerCase().lastIndexOf("green");
                        int blueIndex = value.toLowerCase().lastIndexOf("blue");
                        int index = redIndex > 0 && redIndex > greenIndex && redIndex > blueIndex ? redIndex + "red".length() : (greenIndex > 0 && greenIndex > redIndex && greenIndex > blueIndex ? greenIndex + "green".length() : (blueIndex > 0 && blueIndex > redIndex && blueIndex > greenIndex ? blueIndex + "blue".length() : value.indexOf(32)));
                        if (index > 0) {
                            key = key + ' ' + value.substring(0, index);
                            value = value.substring(index + 1);
                        }
                    } else if (key.equalsIgnoreCase("view view color mode rgb set")) {
                        int index = value.toLowerCase().lastIndexOf("colors");
                        index = index > 0 ? (index += "colors".length()) : value.indexOf(32);
                        if (index > 0) {
                            key = key + ' ' + value.substring(0, index);
                            value = value.substring(index + 1);
                        }
                    }
                    this.addGlobalMeta(key, value);
                    this.trackKeyValues(id, key, value, "");
                } else {
                    LOGGER.debug("Unknown category " + token0);
                }
            }
            line = reader.readString(NL);
        }
        reader.close();
        this.hasInstrumentData = emWaves != null || exWaves != null || lensNA != null || stagePos != null || magnification != null || workingDistance != null || objectiveModel != null || immersion != null;
        this.addGlobalMeta("history text", textBlock.toString());
        LOGGER.info("Populating core metadata");
        this.core[0].rgb = false;
        this.core[0].dimensionOrder = "XY";
        this.channelLengths = new Vector();
        this.channelTypes = new Vector();
        int bitsPerPixel = 0;
        for (int i = 0; i < axes.length && i < axisLengths.length; ++i) {
            if (axes[i].equals("bits")) {
                bitsPerPixel = axisLengths[i];
                while (bitsPerPixel % 8 != 0) {
                    ++bitsPerPixel;
                }
                if (bitsPerPixel != 24 && bitsPerPixel != 48) continue;
                bitsPerPixel /= 3;
                continue;
            }
            if (axes[i].equals("x")) {
                this.core[0].sizeX = axisLengths[i];
                continue;
            }
            if (axes[i].equals("y")) {
                this.core[0].sizeY = axisLengths[i];
                continue;
            }
            if (axes[i].equals("z")) {
                this.core[0].sizeZ = axisLengths[i];
                if (this.getDimensionOrder().indexOf("Z") != -1) continue;
                this.core[0].dimensionOrder = this.core[0].dimensionOrder + "Z";
                continue;
            }
            if (axes[i].equals("t")) {
                this.core[0].sizeT = this.getSizeT() == 0 ? axisLengths[i] : (this.core[0].sizeT *= axisLengths[i]);
                if (this.getDimensionOrder().indexOf("T") != -1) continue;
                this.core[0].dimensionOrder = this.core[0].dimensionOrder + "T";
                continue;
            }
            this.core[0].sizeC = this.core[0].sizeC == 0 ? axisLengths[i] : (this.core[0].sizeC *= axisLengths[i]);
            this.channelLengths.add(new Integer(axisLengths[i]));
            this.storedRGB = this.getSizeX() == 0;
            boolean bl = this.core[0].rgb = this.getSizeX() == 0 && this.getSizeC() <= 4 && this.getSizeC() > 1;
            if (this.getDimensionOrder().indexOf("C") == -1) {
                this.core[0].dimensionOrder = this.core[0].dimensionOrder + "C";
            }
            if (axes[i].startsWith("c")) {
                this.channelTypes.add("Channel");
                continue;
            }
            if (axes[i].equals("p")) {
                this.channelTypes.add("Phase");
                continue;
            }
            if (axes[i].equals("f")) {
                this.channelTypes.add("Frequency");
                continue;
            }
            this.channelTypes.add("");
        }
        if (this.channelLengths.size() == 0) {
            this.channelLengths.add(new Integer(1));
            this.channelTypes.add("Channel");
        }
        this.core[0].dimensionOrder = MetadataTools.makeSaneDimensionOrder(this.getDimensionOrder());
        if (this.getSizeZ() == 0) {
            this.core[0].sizeZ = 1;
        }
        if (this.getSizeC() == 0) {
            this.core[0].sizeC = 1;
        }
        if (this.getSizeT() == 0) {
            this.core[0].sizeT = 1;
        }
        this.core[0].interleaved = this.isRGB();
        this.core[0].indexed = false;
        this.core[0].falseColor = false;
        this.core[0].metadataComplete = true;
        this.core[0].littleEndian = true;
        if (this.lifetime && this.labels != null) {
            int binCount = 0;
            String newOrder = null;
            if (this.labels.equalsIgnoreCase("t x y")) {
                newOrder = "XYCZT";
                this.core[0].interleaved = true;
                binCount = this.core[0].sizeX;
                this.core[0].sizeX = this.core[0].sizeY;
                this.core[0].sizeY = this.core[0].sizeZ;
                this.core[0].sizeZ = 1;
            } else if (this.labels.equalsIgnoreCase("x y t")) {
                newOrder = "XYCZT";
                binCount = this.core[0].sizeZ;
                this.core[0].sizeZ = 1;
            } else {
                LOGGER.debug("Lifetime data, unexpected 'history labels' " + this.labels);
            }
            if (newOrder != null) {
                this.core[0].dimensionOrder = newOrder;
                this.core[0].sizeC = binCount;
                this.core[0].cLengths = new int[]{binCount};
                this.core[0].cTypes = new String[]{"Lifetime"};
            }
        }
        this.core[0].imageCount = this.getSizeZ() * this.getSizeT();
        if (!this.isRGB()) {
            this.core[0].imageCount *= this.getSizeC();
        }
        if (byteOrder != null) {
            String firstByte = byteOrder.split(" ")[0];
            int first = Integer.parseInt(firstByte);
            this.core[0].littleEndian = rFormat.equals("real") ? first == 1 : first != 1;
        }
        boolean bl = this.gzip = compression == null ? false : compression.equals("gzip");
        if (this.versionTwo) {
            String s = this.in.readString(NL);
            while (!s.trim().equals("end")) {
                s = this.in.readString(NL);
            }
        }
        this.offset = this.in.getFilePointer();
        int bytes = bitsPerPixel / 8;
        if (bitsPerPixel < 32) {
            this.core[0].littleEndian = !this.isLittleEndian();
        }
        boolean fp = rFormat.equals("real");
        this.core[0].pixelType = FormatTools.pixelTypeFromBytes(bytes, signed, fp);
        LOGGER.info("Populating OME metadata");
        MetadataStore store = this.makeFilterMetadata();
        MetadataTools.populatePixels(store, this, true);
        store.setImageName(imageName, 0);
        if (date != null) {
            store.setImageAcquiredDate(date, 0);
        } else {
            MetadataTools.setDefaultCreationDate(store, id, 0);
        }
        if (this.getMetadataOptions().getMetadataLevel() != MetadataLevel.MINIMUM) {
            int index;
            store.setImageDescription(description, 0);
            String instrumentID = MetadataTools.createLSID("Instrument", 0);
            store.setInstrumentID(instrumentID, 0);
            store.setMicroscopeModel(microscopeModel, 0);
            store.setMicroscopeManufacturer(microscopeManufacturer, 0);
            store.setImageInstrumentRef(instrumentID, 0);
            store.setExperimentID(MetadataTools.createLSID("Experiment", 0), 0);
            store.setExperimentType(this.getExperimentType(experimentType), 0);
            if (pixelSizes != null) {
                for (int i = 0; i < pixelSizes.length; ++i) {
                    String unit;
                    Double pixelSize = pixelSizes[i];
                    String axis = axes != null && axes.length > i ? axes[i] : "";
                    String string2 = unit = units != null && units.length > i ? units[i] : "";
                    if (axis.equals("x")) {
                        if (!(pixelSize > 0.0) || !this.checkUnit(unit, "um", "microns")) continue;
                        store.setPixelsPhysicalSizeX(new PositiveFloat(pixelSize), 0);
                        continue;
                    }
                    if (axis.equals("y")) {
                        if (!(pixelSize > 0.0) || !this.checkUnit(unit, "um", "microns")) continue;
                        store.setPixelsPhysicalSizeY(new PositiveFloat(pixelSize), 0);
                        continue;
                    }
                    if (axis.equals("z")) {
                        if (!(pixelSize > 0.0) || !this.checkUnit(unit, "um", "microns")) continue;
                        store.setPixelsPhysicalSizeZ(new PositiveFloat(pixelSize), 0);
                        continue;
                    }
                    if (!axis.equals("t") || !this.checkUnit(unit, "ms")) continue;
                    store.setPixelsTimeIncrement(1000.0 * pixelSize, 0);
                }
            } else if (sizes != null) {
                if (sizes.length > 0 && sizes[0] > 0.0) {
                    store.setPixelsPhysicalSizeX(new PositiveFloat(Double.valueOf(sizes[0])), 0);
                }
                if (sizes.length > 1) {
                    sizes[1] = sizes[1] / (double)this.getSizeY();
                    if (sizes[1] > 0.0) {
                        store.setPixelsPhysicalSizeY(new PositiveFloat(Double.valueOf(sizes[1])), 0);
                    }
                }
            }
            if (timestamps != null) {
                for (int t = 0; t < timestamps.length && t < this.getSizeT(); ++t) {
                    double deltaT;
                    if (timestamps[t] == null || Double.isNaN(deltaT = timestamps[t].doubleValue())) continue;
                    for (int z = 0; z < this.getSizeZ(); ++z) {
                        for (int c2 = 0; c2 < this.getEffectiveSizeC(); ++c2) {
                            index = this.getIndex(z, c2, t);
                            store.setPlaneDeltaT(deltaT, 0, index);
                        }
                    }
                }
            }
            for (int i = 0; i < this.getEffectiveSizeC(); ++i) {
                if (channelNames.containsKey(i)) {
                    store.setChannelName((String)channelNames.get(i), 0, i);
                }
                if (pinholes.containsKey(i)) {
                    store.setChannelPinholeSize((Double)pinholes.get(i), 0, i);
                }
                if (emWaves != null && i < emWaves.length && emWaves[i] > 0) {
                    store.setChannelEmissionWavelength(new PositiveInteger(emWaves[i]), 0, i);
                }
                if (exWaves == null || i >= exWaves.length || exWaves[i] <= 0) continue;
                store.setChannelExcitationWavelength(new PositiveInteger(exWaves[i]), 0, i);
            }
            Object[] lasers = wavelengths.keySet().toArray(new Integer[0]);
            Arrays.sort(lasers);
            for (int i = 0; i < lasers.length; ++i) {
                store.setLaserID(MetadataTools.createLSID("LightSource", 0, i), 0, i);
                store.setLaserWavelength(new PositiveInteger((Integer)wavelengths.get(lasers[i])), 0, i);
                store.setLaserType(this.getLaserType("Other"), 0, i);
                store.setLaserLaserMedium(this.getLaserMedium("Other"), 0, i);
                store.setLaserManufacturer(laserManufacturer, 0, i);
                store.setLaserModel(laserModel, 0, i);
                store.setLaserPower(laserPower, 0, i);
                store.setLaserRepetitionRate(laserRepetitionRate, 0, i);
            }
            if (lasers.length == 0 && laserManufacturer != null) {
                store.setLaserID(MetadataTools.createLSID("LightSource", 0, 0), 0, 0);
                store.setLaserType(this.getLaserType("Other"), 0, 0);
                store.setLaserLaserMedium(this.getLaserMedium("Other"), 0, 0);
                store.setLaserManufacturer(laserManufacturer, 0, 0);
                store.setLaserModel(laserModel, 0, 0);
                store.setLaserPower(laserPower, 0, 0);
                store.setLaserRepetitionRate(laserRepetitionRate, 0, 0);
            }
            if (filterSetModel != null) {
                store.setFilterSetID(MetadataTools.createLSID("FilterSet", 0, 0), 0, 0);
                store.setFilterSetModel(filterSetModel, 0, 0);
                String dichroicID = MetadataTools.createLSID("Dichroic", 0, 0);
                String emFilterID = MetadataTools.createLSID("Filter", 0, 0);
                String exFilterID = MetadataTools.createLSID("Filter", 0, 1);
                store.setDichroicID(dichroicID, 0, 0);
                store.setDichroicModel(dichroicModel, 0, 0);
                store.setFilterSetDichroicRef(dichroicID, 0, 0);
                store.setFilterID(emFilterID, 0, 0);
                store.setFilterModel(emissionModel, 0, 0);
                store.setFilterSetEmissionFilterRef(emFilterID, 0, 0, 0);
                store.setFilterID(exFilterID, 0, 1);
                store.setFilterModel(excitationModel, 0, 1);
                store.setFilterSetExcitationFilterRef(exFilterID, 0, 0, 0);
            }
            if (objectiveModel != null) {
                store.setObjectiveModel(objectiveModel, 0, 0);
            }
            if (immersion == null) {
                immersion = "Other";
            }
            store.setObjectiveImmersion(this.getImmersion(immersion), 0, 0);
            if (lensNA != null) {
                store.setObjectiveLensNA(lensNA, 0, 0);
            }
            if (workingDistance != null) {
                store.setObjectiveWorkingDistance(workingDistance, 0, 0);
            }
            if (magnification != null) {
                store.setObjectiveCalibratedMagnification(magnification, 0, 0);
            }
            store.setObjectiveCorrection(this.getCorrection("Other"), 0, 0);
            String objectiveID = MetadataTools.createLSID("Objective", 0, 0);
            store.setObjectiveID(objectiveID, 0, 0);
            store.setImageObjectiveSettingsID(objectiveID, 0);
            String detectorID = MetadataTools.createLSID("Detector", 0, 0);
            store.setDetectorID(detectorID, 0, 0);
            store.setDetectorManufacturer(detectorManufacturer, 0, 0);
            store.setDetectorModel(detectorModel, 0, 0);
            store.setDetectorType(this.getDetectorType("Other"), 0, 0);
            for (Integer key : gains.keySet()) {
                index = key;
                if (index >= this.getEffectiveSizeC()) continue;
                store.setDetectorSettingsGain((Double)gains.get(key), 0, index);
                store.setDetectorSettingsID(detectorID, 0, index);
            }
            if (lastName != null) {
                String experimenterID = MetadataTools.createLSID("Experimenter", 0);
                store.setExperimenterID(experimenterID, 0);
                store.setExperimenterLastName(lastName, 0);
                store.setExperimenterDisplayName(lastName, 0);
            }
            if (stagePos != null) {
                for (int i = 0; i < this.getImageCount(); ++i) {
                    if (stagePos.length > 0) {
                        store.setPlanePositionX(stagePos[0], 0, i);
                        this.addGlobalMeta("X position for position #1", stagePos[0]);
                    }
                    if (stagePos.length > 1) {
                        store.setPlanePositionY(stagePos[1], 0, i);
                        this.addGlobalMeta("Y position for position #1", stagePos[1]);
                    }
                    if (stagePos.length <= 2) continue;
                    store.setPlanePositionZ(stagePos[2], 0, i);
                    this.addGlobalMeta("Z position for position #1", stagePos[2]);
                }
            }
            if (exposureTime != null) {
                for (int i = 0; i < this.getImageCount(); ++i) {
                    store.setPlaneExposureTime(exposureTime, 0, i);
                }
            }
        }
    }

    private String[] tokenize(String line) {
        ArrayList<String> tokens = new ArrayList<String>();
        boolean inWhiteSpace = true;
        boolean withinQuotes = false;
        StringBuffer token = null;
        for (int i = 0; i < line.length(); ++i) {
            char c = line.charAt(i);
            if (Character.isWhitespace(c) || c == '\u0004') {
                if (withinQuotes) {
                    token.append(c);
                    continue;
                }
                if (inWhiteSpace) continue;
                inWhiteSpace = true;
                if (token.length() <= 0) continue;
                tokens.add(token.toString());
                token = null;
                continue;
            }
            if ('\"' == c) {
                boolean bl = withinQuotes = !withinQuotes;
            }
            if (inWhiteSpace) {
                inWhiteSpace = false;
                token = new StringBuffer();
            }
            token.append(c);
        }
        if (null != token && token.length() > 0) {
            tokens.add(token.toString());
        }
        return tokens.toArray(new String[0]);
    }

    String[] findKeyValue(String[] tokens, String[][] regexesArray) {
        String[] keyValue = this.findKeyValueForCategory(tokens, regexesArray);
        if (null == keyValue) {
            keyValue = this.findKeyValueOther(tokens, OTHER_KEYS);
        }
        if (null == keyValue) {
            String key = tokens[0];
            String value = this.concatenateTokens(tokens, 1, tokens.length);
            keyValue = new String[]{key, value};
        }
        return keyValue;
    }

    private String concatenateTokens(String[] tokens, int start, int stop) {
        StringBuffer returnValue = new StringBuffer();
        for (int i = start; i < tokens.length && i < stop; ++i) {
            returnValue.append(tokens[i]);
            if (i >= stop - 1) continue;
            returnValue.append(' ');
        }
        return returnValue.toString();
    }

    private String[] findKeyValueForCategory(String[] tokens, String[][] regexesArray) {
        String[] keyValue = null;
        int index = 0;
        for (String[] regexes : regexesArray) {
            if (this.compareTokens(tokens, 1, regexes, 0)) {
                int splitIndex = 1 + regexes.length;
                String key = this.concatenateTokens(tokens, 0, splitIndex);
                String value = this.concatenateTokens(tokens, splitIndex, tokens.length);
                keyValue = new String[]{key, value};
                break;
            }
            ++index;
        }
        return keyValue;
    }

    private String[] findKeyValueOther(String[] tokens, String[][] regexesArray) {
        String[] keyValue = null;
        for (String[] regexes : regexesArray) {
            for (int i = 1; i < tokens.length - regexes.length; ++i) {
                if (!tokens[i].toLowerCase().matches(regexes[0]) || 1 != regexes.length && !this.compareTokens(tokens, i + 1, regexes, 1)) continue;
                int splitIndex = i + regexes.length;
                String key = this.concatenateTokens(tokens, 0, splitIndex);
                String value = this.concatenateTokens(tokens, splitIndex, tokens.length);
                keyValue = new String[]{key, value};
                break;
            }
            if (null != keyValue) break;
        }
        return keyValue;
    }

    private boolean compareTokens(String[] tokens, int tokenIndex, String[] regexes, int regexesIndex) {
        boolean returnValue = true;
        int i = tokenIndex;
        for (int j = regexesIndex; j < regexes.length; ++j) {
            if (i >= tokens.length || !tokens[i].toLowerCase().matches(regexes[j])) {
                returnValue = false;
                break;
            }
            ++i;
        }
        return returnValue;
    }

    private Double[] splitDoubles(String v) {
        StringTokenizer t = new StringTokenizer(v);
        Double[] values = new Double[t.countTokens()];
        for (int n = 0; n < values.length; ++n) {
            String token = t.nextToken().trim();
            try {
                values[n] = new Double(token);
                continue;
            }
            catch (NumberFormatException e) {
                LOGGER.debug("Could not parse double value '{}'", (Object)token, (Object)e);
            }
        }
        return values;
    }

    private boolean checkUnit(String actual, String ... expected) {
        if (actual == null || actual.equals("")) {
            return true;
        }
        for (String exp : expected) {
            if (!actual.equals(exp)) continue;
            return true;
        }
        LOGGER.debug("Unexpected unit '{}'; expected '{}'", (Object)actual, (Object)expected);
        return false;
    }

    private void trackKeyValues(String id, String key, String value, String parent) {
        TrackKeyValueClass keyValueClass = this.keyValueMap.get(key = key.toLowerCase());
        if (null == keyValueClass) {
            keyValueClass = new TrackKeyValueClass(id, value, parent);
            this.keyValueMap.put(key, keyValueClass);
        } else {
            keyValueClass.values.add(value);
            keyValueClass.ids.add(id);
        }
    }

    private void dumpKeyValueClasses(String rootDir) {
        Set<String> keys = this.keyValueMap.keySet();
        ArrayList<String> keyList = new ArrayList<String>(keys);
        Collections.sort(keyList);
        for (String key : keyList) {
            TrackKeyValueClass keyValueClass = this.keyValueMap.get(key);
            System.out.print(key);
            for (String parent : keyValueClass.parents) {
                System.out.print(" " + parent);
            }
            System.out.println();
            for (String value : keyValueClass.values) {
                System.out.println("  " + value);
            }
            for (String id : keyValueClass.ids) {
                id = id.substring(rootDir.length());
                System.out.print(" " + id);
            }
            System.out.println();
        }
    }

    private void dumpSortedKeys() {
        System.out.println();
        System.out.println();
        Set<String> keys = this.keyValueMap.keySet();
        HashSet<String> lowercaseKeys = new HashSet<String>();
        for (String key : keys) {
            lowercaseKeys.add(key.toLowerCase());
        }
        ArrayList keyList = new ArrayList(lowercaseKeys);
        Collections.sort(keyList);
        for (String key : keyList) {
            String[] subkeys = key.split(" ");
            boolean first = true;
            for (String subkey : subkeys) {
                if (first) {
                    System.out.print("    { ");
                    first = false;
                } else {
                    System.out.print(", ");
                }
                System.out.print('\"' + subkey.toLowerCase() + '\"');
            }
            System.out.println(" },");
        }
        System.out.println();
        System.out.println();
    }

    private void recursiveSearch(boolean old, File dir) {
        File[] files = dir.listFiles();
        if (null != files) {
            for (File f : files) {
                if (f.isDirectory()) {
                    this.recursiveSearch(old, f);
                    continue;
                }
                String id = f.getAbsolutePath();
                if (!id.toLowerCase().endsWith(".ids") && !id.toLowerCase().endsWith(".ics")) continue;
                try {
                    System.out.println(id);
                    ++this.count;
                    if (old) {
                        this.oldInitFile(id);
                        continue;
                    }
                    this.initFile(id);
                }
                catch (Exception e) {
                    System.out.println("Exception processing " + id + " " + e.getMessage());
                }
            }
        }
    }

    public static void main(String[] args) {
        String rootDir = "/users/aivar/Desktop/ics";
        if (args.length > 0) {
            rootDir = args[0];
        }
        File root = new File(rootDir);
        long startTime = System.currentTimeMillis();
        ICSReader reader = new ICSReader();
        reader.recursiveSearch(true, root);
        reader.dumpKeyValueClasses(rootDir);
        reader.dumpSortedKeys();
        System.out.println("Files processed: " + reader.count);
        System.out.println("elapsed time " + (System.currentTimeMillis() - startTime));
        System.out.println("-------------------------------------");
        startTime = System.currentTimeMillis();
        reader = new ICSReader();
        reader.recursiveSearch(false, root);
        reader.dumpKeyValueClasses(rootDir);
        reader.dumpSortedKeys();
        System.out.println("Files processed: " + reader.count);
        System.out.println("elapsed time " + (System.currentTimeMillis() - startTime));
    }

    private class TrackKeyValueClass {
        List<String> values = new ArrayList<String>();
        List<String> ids;
        Set<String> parents;

        TrackKeyValueClass(String id, String value, String parent) {
            this.values.add(value);
            this.ids = new ArrayList<String>();
            this.ids.add(id);
            this.parents = new HashSet<String>();
            this.parents.add(parent);
        }
    }
}

